<?php

declare(strict_types=1);

namespace Studio148\ProlesApi;

use Illuminate\Http\Client\PendingRequest;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Spatie\Permission\Models\Role;
use App\Models\User;
use Studio148\ProlesApi\Contracts\ProlesApiContract;
use Studio148\ProlesApi\Exceptions\ProlesApiExceptionHandler;

class ProlesApi implements ProlesApiContract
{
    protected PendingRequest|Response $client;

    public function __construct()
    {
        $this->client = Http::withOptions([
            'base_uri' => config('proles_api.url')]);
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function authenticate(string $username, string $password): mixed
    {
        $response = $this->sendRequestToProlesApi('check_login_gebruiker', [
            'email' => $username,
            'pass' => $password,
        ]);

        return $response->data->hash;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function authenticateUsingHashedPassword(string $username, string $password): mixed
    {
        $response = $this->sendRequestToProlesApi('check_login_gebruiker', [
            'email' => $username,
            'pass' => md5($password),
        ]);

        return $response->data->hash;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function getResetPasswordTemporaryPassword(string $username): string
    {
        $url = 'gegevens/ophalen/medewerker/wwvergeten';

        $user = User::where('username', $username)->first();
        if($user->roles->contains('alias', 'ouders')) $url = 'gegevens/ophalen/contact/wwvergeten';

        return $this->sendRequestToProlesApi($url, [
            'email' => $username,
            'temppass' => '',
        ])->data->temppass;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function resetPassword($username, $token, $password): object|string
    {
        $url = 'gegevens/ophalen/medewerker/wwvergeten';

        $user = User::where('username', $username)->first();
        if($user->roles->contains('alias', 'ouders')) $url = 'gegevens/ophalen/contact/wwvergeten';

        return $this->sendRequestToProlesApi($url, [
            'email' => $username,
            'temppass' => $token,
            'newpass' => $password
        ]);
    }

    /** Get all contact and related data for parents
     * @throws ProlesApiExceptionHandler
     */
    public function getContact(string $email, string $hash): mixed
    {
        $response = $this->sendRequestToProlesApi('gegevens/ophalen/contact', [
            'email' => $email,
            'hash' => $hash
        ]);

        return $response->data;
    }

    /** Update parent contact data
     * @throws ProlesApiExceptionHandler
     */
    public function updateContact(string $email, string $hash, array $parameters): \stdClass
    {
        $response = $this->sendRequestToProlesApi('gegevens/wijzigen/contact', [
            'email' => $email,
            'hash' => $hash,
            'naamOuders' => $parameters['naamOuders'],
            'emailOuders' => $parameters['email'],
            'telefoonPrive' => $parameters['telefoonPrive'],
            'telefoonWerk' => $parameters['telefoonWerk'],
            'telefoonMobiel' => $parameters['telefoonMobiel']
        ]);

        return $response->data;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function addChildToFamily(User $user, \stdClass $child, string $hash): \stdClass
    {
        return $this->sendRequestToProlesApi('gegevens/inschrijven/extra/kind', [
            'voornaam' => $child->voornaam,
            'tussenvoegsels' => $child->tussenvoegsels,
            'achternaam' => $child->achternaam,
            'geslacht' => $child->geslacht,
            'geboortedatum' => $child->geboortedatum,
            'bijzonderheden' => $child->bijzonderheden,
            'schoolId' => $child->schoolId,
            'datumStart' => $child->datumStart,
            'groepId' =>  $child->groepId,
            'ma' => $child->opvang_ma ? 'j' : 'n',
            'di' => $child->opvang_di ? 'j' : 'n',
            'wo' => $child->opvang_wo ? 'j' : 'n',
            'do' => $child->opvang_do ? 'j' : 'n',
            'vr' => $child->opvang_vr ? 'j' : 'n',
            'gezin_id' => $user->gezin_id,
            'email' => $user->username,
            'hash' => $hash
        ])->data;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function updateChild($email, $hash, $kindId, $kindgegevens): \StdClass
    {
        $response = $this->sendRequestToProlesApi('gegevens/wijzigen/kind', [
            'email' => $email,
            'hash' => $hash,
            'kind_id' => $kindId,
            'voornaam' => $kindgegevens['voornaam'],
            'tussenvoegsels' => $kindgegevens['tussenvoegsels'],
            'achternaam' => $kindgegevens['achternaam'],
            'geslacht' => $kindgegevens['geslacht'],
            'geboortedatum' => $kindgegevens['geboortedatum'],
            'adres' => $kindgegevens['adres'],
            'postcode' => $kindgegevens['postcode'],
            'woonplaats' => $kindgegevens['woonplaats'],
            'bijzonderheden' => $kindgegevens['bijzonderheden'],
        ]);

        return $response->data;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function updatePaymentInfo(User $user, string $hash): \stdClass
    {
        $response = $this->sendRequestToProlesApi('gegevens/inschrijven/betaalwijze', [
            'email' => $user->email,
            'hash' => $hash,
            'id' => $user->remote_id,
            'bic' => $user->bic,
            'iban' => $user->iban,
            'rekeninghouder' => $user->rekeninghouder,
            'sepaDatum' => $user->sepaDatum,
            'sepaMandaat' => $user->sepaMandaat
        ]);

        return $response->data;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function getStructureleOpvangDagenForChild(string $email, string $hash, int $kindId): array
    {
        $response = $this->sendRequestToProlesApi('gegevens/ophalen/structureel', [
            'email' => $email,
            'hash' => $hash,
            'kind_id' => $kindId,
        ]);

        return $response->data->opvangdagen;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function updateStructureleOpvangDagenForChild($email, $hash, $kindId, $startDate, $newContractDays): array
    {
        $response = $this->sendRequestToProlesApi('gegevens/ophalen/structureel', [
            'email' => $email,
            'hash' => $hash,
            'kind_id' => $kindId,
            'datum' => $startDate,
            'ma' => array_key_exists('ma', $newContractDays) ? 'j' : 'n',
            'di' => array_key_exists('di', $newContractDays) ? 'j' : 'n',
            'wo' => array_key_exists('wo', $newContractDays) ? 'j' : 'n',
            'do' => array_key_exists('do', $newContractDays) ? 'j' : 'n',
            'vr' => array_key_exists('vr', $newContractDays) ? 'j' : 'n',
        ]);

        return $response->data->opvangdagen;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function getPresenceForChild(string $email, string $hash, int $kindId): \stdClass
    {
        $response = $this->sendRequestToProlesApi('gegevens/ophalen/presentie', [
            'email' => $email,
            'hash' => $hash,
            'kind_id' => $kindId,
        ]);

        return $response->data;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function updatePresenceForChild($email, $hash, $kindId, $datum): \stdClass
    {
        $response = $this->sendRequestToProlesApi('gegevens/ophalen/presentie', [
            'email' => $email,
            'hash' => $hash,
            'kind_id' => $kindId,
            'datum' => $datum,
        ]);

        return $response->data;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function getExtraDataForCoordinator(string $email, string $hash): mixed
    {
        $response = $this->sendRequestToProlesApi('gegevens/ophalen/coordinator', [
            'email' => $email,
            'hash' => $hash
        ]);

        return $response->data->coordinator;
    }

    /**
     * @throws ProlesApiExceptionHandler
     */
    public function getExtraDataForOverblijfmedewerker(string $email, string $hash): mixed
    {
        $response = $this->sendRequestToProlesApi('gegevens/ophalen/vrijwilliger', [
            'email' => $email,
            'hash' => $hash
        ]);

        return $response->data->vrijwilliger;
    }

    /** Previously named getSchools
     * @throws ProlesApiExceptionHandler
     */
    public function getSchoolsFromProlesApi(): mixed
    {
        $response = $this->sendRequestToProlesApi('gegevens/ophalen/scholen/sync', [
            'email' => Config('proles_api.sync_user'),
            'pass' => Config('proles_api.sync_pass')]);

        return $response->data->scholen;
    }

    /** Previously named getUsers
     * @throws ProlesApiExceptionHandler
     */
    public function getUsersFromProlesApi(): Collection
    {
        $response = $this->sendRequestToProlesApi('gegevens/ophalen/gebruikers/sync', [
            'email' => Config('proles_api.sync_user'),
            'pass' => Config('proles_api.sync_pass'),
        ]);

        $users = collect();

        $ouderRole = Role::where('alias', 'ouders')->first();
        $overblijfMedewerkerRole = Role::where('alias', 'overblijfmedewerkers')->first();
        $overblijfCoordinatorRole = Role::where('alias', 'overblijfcoordinatoren')->first();
        $regiomanagerRole = Role::where('alias', 'regiomanagers')->first();
        $schoolRole = Role::where('alias', 'schoolmedewerkers')->first();
        $centraalBureauRole = Role::where('alias', 'centraal-bureau')->first();

        $data = $response->data;

        foreach (['gebruikers', 'vrijwilligers', 'coordinatoren', 'regiomanagers', 'schooldirecteuren', 'medewerkers'] as $remoteUserType) {

            foreach ($data->$remoteUserType as $user) {
                if (! empty($user->gebruikersnaam)) {
                    $user->type = $remoteUserType;
                    $user->remoteId = $remoteUserType . '-' . $user->guid;
                    switch ($remoteUserType) {
                        case 'gebruikers':
                            $user->groupId = $ouderRole->id;
                            break;
                        case 'vrijwilligers':
                            $user->groupId = $overblijfMedewerkerRole->id;
                            break;
                        case 'coordinatoren':
                            $user->groupId = $overblijfCoordinatorRole->id;
                            break;
                        case 'regiomanagers':
                            $user->groupId = $regiomanagerRole->id;
                            break;
                        case 'schooldirecteuren':
                            $user->groupId = $schoolRole->id;
                            break;
                        case 'medewerkers':
                            $user->groupId = $centraalBureauRole->id;
                            break;
                        default:
                            throw new \Exception('Unexpected value');
                    }
                    $users->push($user);
                }
            }
        }

        return $users;
    }

    /** Previously named send
     * @throws ProlesApiExceptionHandler
     */
    private function sendRequestToProlesApi(string $url, array $parameters = [], string $method = 'post'): string|object
    {
        $parameters = $this->sanitizeRequestParameters($parameters);

        if (! array_key_exists('md5hash', $parameters)) {
            $parameters = array_merge($parameters, ['md5hash' => $this->generateMd5Hash($parameters)]);
        }

        $response = $this->client->{$method}($url, $parameters);

        if ($response->successful()) {
            $json = json_decode($response->body());
            if ($json !== null) {
                return $json;
            }

            return $response->body();
        }

        if ($response->clientError()) {
            throw ProlesApiExceptionHandler::IncorrectMD5HashException($response);
        }

        if ($response->serverError()) {
            throw ProlesApiExceptionHandler::ServerErrorException($response);
        }
    }

    /** Previously named getMd5 */
    private function generateMd5Hash(array $array): string
    {
        $parameters = [];
        foreach ($array as $parameter) {
            if (is_array($parameter)) {
                $parameters[] = 'Array';
            } else {
                $parameters[] = trim($parameter);
            }
        }

        return md5($this->getClientServerIp() . base64_encode(implode('', $parameters)));
    }

    /** Previously named sanitize */
    private function sanitizeRequestParameters(array $parameters): array
    {
        $sanitized = [];

        foreach ($parameters as $key => $value) {
            if (is_null($value)) {
                $sanitized[$key] = null;
                continue;
            }
            $sanitized[$key] = is_array($value) ? $value : str_replace("'", '`', (string) $value);
        }

        return $sanitized;
    }

    /** Previously named getServerIp */
    private function getClientServerIp(): string
    {
        return env('SERVER_IP');
    }
}
