<?php

declare(strict_types=1);

namespace Studio148\ProlesApi;

use Illuminate\Http\Client\PendingRequest;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Spatie\Permission\Models\Role;

class ProlesApi
{
    protected PendingRequest|Response $client;

    public function __construct()
    {
        $this->client = Http::withOptions([
            'base_uri' => config('proles_api.url')]);
    }

    /** Previously named getSchools
     * @throws ProlesApiExceptionHandler
     */
    public function getSchoolsFromProlesApi(): mixed
    {
        $response = $this->sendRequestToProlesApi('gegevens/ophalen/scholen/sync', [
            'email' => Config('proles_api.sync_user'),
            'pass' => Config('proles_api.sync_pass')]);

        return $response->data->scholen;
    }

    /** Previously named getUsers
     * @throws ProlesApiExceptionHandler
     */
    public function getUsersFromProlesApi(): Collection
    {
        $response = $this->sendRequestToProlesApi('gegevens/ophalen/gebruikers/sync', [
            'email' => Config('proles_api.sync_user'),
            'pass' => Config('proles_api.sync_pass'),
        ]);

        $users = collect();

        $ouderRole = Role::where('alias', 'ouders')->first();
        $overblijfMedewerkerRole = Role::where('alias', 'overblijfmedewerkers')->first();
        $overblijfCoordinatorRole = Role::where('alias', 'overblijfcoordinatoren')->first();
        $regiomanagerRole = Role::where('alias', 'regiomanagers')->first();
        $schoolRole = Role::where('alias', 'schoolmedewerkers')->first();
        $centraalBureauRole = Role::where('alias', 'centraal-bureau')->first();

        $data = $response->data;

        foreach (['gebruikers', 'vrijwilligers', 'coordinatoren', 'regiomanagers', 'schooldirecteuren', 'medewerkers'] as $remoteUserType) {

            foreach ($data->$remoteUserType as $user) {
                if (! empty($user->gebruikersnaam)) {
                    $user->type = $remoteUserType;
                    $user->remoteId = $remoteUserType . '-' . $user->guid;
                    switch ($remoteUserType) {
                        case 'gebruikers':
                            $user->groupId = $ouderRole->id;
                            break;
                        case 'vrijwilligers':
                            $user->groupId = $overblijfMedewerkerRole->id;
                            break;
                        case 'coordinatoren':
                            $user->groupId = $overblijfCoordinatorRole->id;
                            break;
                        case 'regiomanagers':
                            $user->groupId = $regiomanagerRole->id;
                            break;
                        case 'schooldirecteuren':
                            $user->groupId = $schoolRole->id;
                            break;
                        case 'medewerkers':
                            $user->groupId = $centraalBureauRole->id;
                            break;
                        default:
                            throw new \Exception('Unexpected value');
                    }
                    $users->push($user);
                }
            }
        }

        return $users;
    }

    /** Previously named send
     * @throws ProlesApiExceptionHandler
     */
    private function sendRequestToProlesApi(string $url, array $parameters = [], string $method = 'post'): string|object
    {
        $parameters = $this->sanitizeRequestParameters($parameters);

        if (! array_key_exists('md5hash', $parameters)) {
            $parameters = array_merge($parameters, ['md5hash' => $this->generateMd5Hash($parameters)]);
        }

        $response = $this->client->{$method}($url, $parameters);

        if ($response->successful()) {
            $json = json_decode($response->body());
            if ($json !== null) {
                return $json;
            }

            return $response->body();
        }

        if ($response->clientError()) {
            throw ProlesApiExceptionHandler::IncorrectMD5HashException($response);
        }

        if ($response->serverError()) {
            throw ProlesApiExceptionHandler::ServerErrorException($response);
        }
    }

    /** Previously named getMd5 */
    private function generateMd5Hash(array $array): string
    {
        $parameters = [];
        foreach ($array as $parameter) {
            if (is_array($parameter)) {
                $parameters[] = 'Array';
            } else {
                $parameters[] = trim($parameter);
            }
        }

        return md5($this->getClientServerIp() . base64_encode(implode('', $parameters)));
    }

    /** Previously named sanitize */
    private function sanitizeRequestParameters($parameters): array
    {
        $sanitized = [];

        foreach ($parameters as $key => $value) {
            $sanitized[$key] = is_array($value) ? $value : str_replace("'", '`', $value);
        }

        return $sanitized;
    }

    /** Previously named getServerIp */
    private function getClientServerIp(): string
    {
        return env('SERVER_IP');
    }
}
